﻿using System.Collections.Generic;
using RimWorld;
using Verse;
using Verse.AI;

namespace PickUpAndHaul
{
    public class PawnUnloadChecker
    {
        public static void CheckIfPawnShouldUnloadInventory(Pawn pawn, bool forced = false)
        {
            Job job = JobMaker.MakeJob(PickUpAndHaulJobDefOf.UnloadYourHauledInventory, pawn);
            CompHauledToInventory itemsTakenToInventory = pawn.TryGetComp<CompHauledToInventory>();

            if (itemsTakenToInventory == null)
                return;

            HashSet<Thing> carriedThing = itemsTakenToInventory.GetHashSet();

            if (pawn.Faction != Faction.OfPlayer || !pawn.RaceProps.Humanlike)
                return;
            if (carriedThing == null || carriedThing.Count == 0 || pawn.inventory.innerContainer.Count == 0)
                return;

            if (forced)
            {
                if (job.TryMakePreToilReservations(pawn, false))
                {
                    pawn.jobs.jobQueue.EnqueueFirst(job, JobTag.Misc);
                    return;
                }
            }

            if (MassUtility.EncumbrancePercent(pawn) >= 0.90f || carriedThing.Count >= 1)
            {
                if (job.TryMakePreToilReservations(pawn, false))
                {
                    pawn.jobs.jobQueue.EnqueueFirst(job, JobTag.Misc);
                    return;
                }
            }

            if (pawn.inventory.innerContainer?.Count >= 1)
            {
                foreach (Thing rottable in pawn.inventory.innerContainer)
                {
                    CompRottable compRottable = rottable.TryGetComp<CompRottable>();

                    if (compRottable?.TicksUntilRotAtCurrentTemp < 30000)
                    {
                        pawn.jobs.jobQueue.EnqueueFirst(job, JobTag.Misc);
                        return;
                    }
                }
            }

            if (Find.TickManager.TicksGame % 50 == 0 && pawn.inventory.innerContainer.Count < carriedThing.Count)
            {
                Verse.Log.Warning("[PickUpAndHaul] " + pawn + " inventory was found out of sync with haul index. Pawn will drop their inventory.");
                carriedThing.Clear();
                pawn.inventory.UnloadEverything = true;
            }
        }
    }

    [DefOf]
    public static class PickUpAndHaulJobDefOf
    {
        public static JobDef UnloadYourHauledInventory;
        public static JobDef HaulToInventory;
    }
}